/*
 * Copyright (C) 2000 Robert Fitzsimons
 */

#include "types.h"
#include "util.h"
#include "unicode.h"
#include "class.h"
#include "classcode.h"
#include "classdata.h"
#include "classinfo.h"
#include <stdarg.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/wait.h>

#define local_to_offset(i, md) ({ \
	kuint __o = 0; \
	kuint __i = (i); \
	kuint __pc = ((md)->flags & FLAG_SIZE_MASK); \
	if(__i < __pc) { \
		__o = ((__pc + 4) - __i) * sizeof(kuint); \
	} else { \
		__o = -(((__i + 1) - __pc) * sizeof(kuint)); \
	} \
	__o; \
})

void init_classcode() {
}

struct opcode {
	kuint8* name;
	void (*handler)(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file);
};

struct opcode opcode_table[];

kuint64 invoke_method(Environment* env, void* method_code, Class* class, ...) {
	asm(
		"\tmovl\t%%ebp, %%esp\n"
		"\tpop\t%%ebp\n"
		"\tjmp\t*%0\n"
		:
		: /* 0 */ "a" (method_code)
	);
	return 0;
}

void bytecode_to_nativecode(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata) {
	kuint8* code = (void*)methoddata->code;
	kuint code_length = methoddata->code_length;
	kuint code_index = 0;
	kuint opcode = 0;
	kuint index = 0;
	kuint8* filename = memory_allocate(36, sizeof(kuint8));
	kuint8* filename_o = memory_allocate(36, sizeof(kuint8));
	FILE* file = knull;
	kuint8* ascii = knull;

	sprintf(filename, "work/%08X_%08X_%08X.s", (unsigned int)classdata->name_handle, (unsigned int)methoddata->name_handle, (unsigned int)methoddata->descriptor_handle);
	sprintf(filename_o, "work/%08X_%08X_%08X.o", (unsigned int)classdata->name_handle, (unsigned int)methoddata->name_handle, (unsigned int)methoddata->descriptor_handle);
	file = fopen(filename, "wb");

	fprintf(file, "/*\n");
	ascii = unicode_get_ascii(classdata->name_handle);
	fprintf(file, " * %08X %s\n", (unsigned int)classdata->name_handle, ascii);
	memory_free(ascii);
	ascii = unicode_get_ascii(methoddata->name_handle);
	fprintf(file, " * %08X %s\n", (unsigned int)methoddata->name_handle, ascii);
	memory_free(ascii);
	ascii = unicode_get_ascii(methoddata->descriptor_handle);
	fprintf(file, " * %08X %s\n", (unsigned int)methoddata->descriptor_handle, ascii);
	memory_free(ascii);
	fprintf(file, " * %08X %04X %04X\n", (unsigned int)methoddata->flags, (unsigned int)methoddata->max_locals, methoddata->max_stack);
	fprintf(file, " * %08X\n", (unsigned int)methoddata->code_length);
	while(index < methoddata->code_length) {
		kuint mod = index % 16;
		if(mod == 0) {
			fprintf(file, " * %08X  ", (unsigned int)index);
		}
		fprintf(file, "%02X", (unsigned char)code[index]);
		index++;
		if((mod == 15) || (index >= code_length)) {
			fprintf(file, "\n");
		} else if(mod == 7) {
			fprintf(file, "  ");
		} else {
			fprintf(file, " ");
		}
	}
	fprintf(file, " */\n\n");

	fprintf(file, ".global\t_start\n");
	fprintf(file, "_start:\n");
	fprintf(file, "\tpushl\t%%ebp\n");
	fprintf(file, "\tmovl\t%%esp, %%ebp\n");

	if((methoddata->max_locals - (methoddata->flags & FLAG_SIZE_MASK)) > 0) {
		fprintf(file, "\tsubl\t$0x%08X, %%esp\n", (unsigned int)(methoddata->max_locals - (methoddata->flags & FLAG_SIZE_MASK)) * 4);
	}

	fprintf(file, "\tpushl\t$0x%08X\n", (unsigned int)methoddata);
	fprintf(file, "\tpushl\t$0x%08X\n", (unsigned int)methoddata->descriptor_handle);
	fprintf(file, "\tpushl\t$0x%08X\n", (unsigned int)methoddata->name_handle);
	fprintf(file, "\tpushl\t$0x%08X\n", (unsigned int)classdata->name_handle);
	fprintf(file, "\tpushl\t$0x%08X\n", (unsigned int)"### %08X %08X %08X 0x%08X ###\n");
	fprintf(file, "\tmovl\t$0x%08X, %%eax\n", (unsigned int)printf);
	fprintf(file, "\tcall\t*%%eax /* call printf */\n");
	fprintf(file, "\taddl\t$20, %%esp\n");

	while(code_index < code_length) {
		opcode = code[code_index];

		if(opcode_table[opcode].handler != knull) {
			fprintf(file, ".L0x%08X: /* %s */\n", (unsigned int)code_index, opcode_table[opcode].name);
			opcode_table[opcode].handler(env, classinfo, classdata, methoddata, code, &code_index, file);
		} else {
			fprintf(file, ".L0x%08X: /* %s FIXME */\n", (unsigned int)code_index, opcode_table[opcode].name);
			fprintf(file, "\tmovl\t$0x00000000, 0x00000BAD\n");
			code_index++;
		}
	}

	fclose(file);

	{
		// Now compile the assembly file into a binary object file
		int status = 0;
		int pid = fork();

		if(pid == 0) {
			execl("/usr/bin/gcc", "gcc", "-static", "--nostartfiles", "-nodefaultlibs", "-nostdlib", "-Wl,--oformat,binary", "-o", filename_o, filename, knull);
		} else if(pid > 0) {
			waitpid(pid, &status, 0); 
		} else {
			abort();
		}
	}

	{
		// Read the binary object file into memory
		kuint8* buffer = knull;
		kuint buffer_length = 0;

		if(read_file(filename_o, &buffer, &buffer_length) == 0) { 
			methoddata->code = (void*)buffer;
			methoddata->code_length = buffer_length;
		} else {
			abort();
		}
	}
}

void opcode_nop(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	read_u1(code, code_index);
}

void opcode_xpush(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);
	kuint value = 0;
	ConstantInfo* ci = knull;

	switch(opcode) {
		case OPCODE_aconst_null: value = (kuint)knull; break;
		case OPCODE_iconst_m1: value = -1; break;
		case OPCODE_iconst_0: value = 0; break;
		case OPCODE_iconst_1: value = 1; break;
		case OPCODE_iconst_2: value = 2; break;
		case OPCODE_iconst_3: value = 3; break;
		case OPCODE_iconst_4: value = 4; break;
		case OPCODE_iconst_5: value = 5; break;
		case OPCODE_fconst_0: value = 0; break;
		case OPCODE_fconst_1: value = 0x3F800000; break;
		case OPCODE_fconst_2: value = 0x40000000; break;
		case OPCODE_bipush: value = read_u1(code, code_index); break;
		case OPCODE_sipush: value = read_u2(code, code_index); break;
		case OPCODE_ldc:
			ci = constant_resolve(env, classinfo, read_u1(code, code_index), 0);
			value = ci->v32_info.bytes;
			break;
		case OPCODE_ldc_w:
			ci = constant_resolve(env, classinfo, read_u2(code, code_index), 0);
			value = ci->v32_info.bytes;
			break;
		default:
			abort();
	}

	fprintf(file, "\tpushl\t$0x%08X\n", (unsigned int)value);
}

void opcode_xload(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);
	kuint index = 0;

	switch(opcode) {
		case OPCODE_iload:
		case OPCODE_aload:
			index = read_u1(code, code_index);
			break;
		case OPCODE_iload_0:
		case OPCODE_aload_0:
			index = 0;
			break;
		case OPCODE_iload_1:
		case OPCODE_aload_1:
			index = 1;
			break;
		case OPCODE_iload_2:
		case OPCODE_aload_2:
			index = 2;
			break;
		case OPCODE_iload_3:
		case OPCODE_aload_3:
			index = 3;
			break;
		default:
			abort();
	}

	fprintf(file, "\tpushl\t0x%08X(%%ebp)\n", (unsigned int)local_to_offset(index, methoddata));
}

void opcode_xstore(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);
	kuint index = 0;

	switch(opcode) {
		case OPCODE_istore:
		case OPCODE_astore:
			index = read_u1(code, code_index);
			break;
		case OPCODE_istore_0:
		case OPCODE_astore_0:
			index = 0;
			break;
		case OPCODE_istore_1:
		case OPCODE_astore_1:
			index = 1;
			break;
		case OPCODE_istore_2:
		case OPCODE_astore_2:
			index = 2;
			break;
		case OPCODE_istore_3:
		case OPCODE_astore_3:
			index = 3;
			break;
		default:
			abort();
	}

	fprintf(file, "\tpopl\t0x%08X(%%ebp)\n", (unsigned int)local_to_offset(index, methoddata));
}

void opcode_stack(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);

	switch(opcode) {
		case OPCODE_pop:
			fprintf(file, "\taddl\t$4, %%esp\n");
			break;
		case OPCODE_pop2:
			fprintf(file, "\taddl\t$8, %%esp\n");
			break;
		case OPCODE_dup:
			fprintf(file, "\tpushl\t0x00000000(%%esp)\n");
			break;
		default:
			abort();
	}
}

void opcode_iop(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);

	if(opcode == OPCODE_ineg) {
		fprintf(file, "\tpopl\t%%eax\n");
		fprintf(file, "\tnegl\t%%eax\n");
		fprintf(file, "\tpushl\t%%eax\n");
	} else {
		fprintf(file, "\tpopl\t%%ecx\n");
		fprintf(file, "\tpopl\t%%eax\n");
		switch(opcode) {
			case OPCODE_iadd:
				fprintf(file, "\taddl\t%%ecx, %%eax\n");
				break;
			case OPCODE_isub:
				fprintf(file, "\tsubl\t%%ecx, %%eax\n");
				break;
			case OPCODE_imul:
				fprintf(file, "\timull\t%%ecx, %%eax\n");
				break;
			case OPCODE_idiv:
			case OPCODE_irem:
				fprintf(file, "\tcltd\n");
				fprintf(file, "\tidivl\t%%ecx, %%eax\n");
				break;
			case OPCODE_ishl:
				fprintf(file, "\tsall\t%%cl, %%eax\n");
				break;
			case OPCODE_ishr:
				fprintf(file, "\tsarl\t%%cl, %%eax\n");
				break;
			case OPCODE_iushr:
				fprintf(file, "\tshrl\t%%cl, %%eax\n");
				break;
			case OPCODE_iand:
				fprintf(file, "\tandl\t%%ecx, %%eax\n");
				break;
			case OPCODE_ior:
				fprintf(file, "\torl\t%%ecx, %%eax\n");
				break;
			case OPCODE_ixor:
				fprintf(file, "\txorl\t%%ecx, %%eax\n");
				break;
			default:
				abort();
		}
		if(opcode != OPCODE_irem) {
			fprintf(file, "\tpushl\t%%eax\n");
		} else {
			fprintf(file, "\tpushl\t%%edx\n");
		}
	}
}

void opcode_iinc(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint index = 0;
	kint value = 0;

	read_u1(code, code_index);
	index = read_u1(code, code_index);
	value = (kint8)read_u1(code, code_index);

	index = local_to_offset(index, methoddata);

	fprintf(file, "\taddl\t$0x%08X, 0x%08X(%%ebp)\n", (unsigned int)value, (unsigned int)index);
}

void opcode_ifx(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint index = (*code_index);
	kuint opcode = read_u1(code, code_index);
	kint offset = (kint16)read_u2(code, code_index);
	kuint8* string = knull;

	fprintf(file, "\tpopl\t%%eax\n");
	fprintf(file, "\tcmpl\t$0, %%eax\n");

	switch(opcode) {
		case OPCODE_ifeq:
		case OPCODE_ifnull:
			string = "je";
			break;
		case OPCODE_ifne:
		case OPCODE_ifnonnull:
			string = "jne";
			break;
		case OPCODE_iflt: string = "jl"; break;
		case OPCODE_ifge: string = "jge"; break;
		case OPCODE_ifgt: string = "jg"; break;
		case OPCODE_ifle: string = "jle"; break;
		default:
			abort();
	}

	fprintf(file, "\t%s\t.L0x%08X\n", string, (unsigned int)(index + offset));
}

void opcode_if_xcmpx(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint index = (*code_index);
	kuint opcode = read_u1(code, code_index);
	kint offset = (kint16)read_u2(code, code_index);
	kuint8* string = knull;

	fprintf(file, "\tpopl\t%%ecx\n");
	fprintf(file, "\tpopl\t%%eax\n");
	fprintf(file, "\tcmpl\t%%ecx, %%eax\n");

	switch(opcode) {
		case OPCODE_if_icmpeq:
		case OPCODE_if_acmpeq:
			string = "je";
			break;
		case OPCODE_if_icmpne:
		case OPCODE_if_acmpne:
			string = "jne";
			break;
		case OPCODE_if_icmplt: string = "jl"; break;
		case OPCODE_if_icmpge: string = "jge"; break;
		case OPCODE_if_icmpgt: string = "jg"; break;
		case OPCODE_if_icmple: string = "jle"; break;
		default:
			abort();
	}

	fprintf(file, "\t%s\t.L0x%08X\n", string, (unsigned int)(index + offset));
}

void opcode_goto(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint index = (*code_index);
	kint offset = 0;

	read_u1(code, code_index);
	offset = (kint16)read_u2(code, code_index);

	fprintf(file, "\tjmp\t.L0x%08X\n", (unsigned int)(index + offset));
}

void opcode_xreturn(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);

	switch(opcode) {
		case OPCODE_ireturn:
		case OPCODE_areturn:
			fprintf(file, "\tpopl\t%%eax\n");
			break;
		case OPCODE_return:
			break;
		default:
			abort();
	}

	fprintf(file, "\tmovl\t%%ebp, %%esp\n");
	fprintf(file, "\tpopl\t%%ebp\n");
	fprintf(file, "\tret\n");
}

void opcode_xfield(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);
	ConstantInfo* fci = constant_resolve(env, classinfo, read_u2(code, code_index), 1);
	ConstantInfo* cci = constant_resolve(env, classinfo, fci->ref.class_index, 1);
	kuint field_flags = 0;
	kuint object_offset = (opcode == OPCODE_getfield) ? (0) : (1);

	parse_field_descriptor(fci->ref.descriptor_handle, &field_flags);

	if((field_flags & FLAG_SIZE_MASK) > 4) {
		abort();
	}

	// FIXME RF 2000-09-09.  There must be a better way to do this.
	fprintf(file, "\tmovl\t0x%08X(%%ebp), %%eax\n", (unsigned int)class_offset);
	fprintf(file, "\tmovl\t0x%08X(%%eax), %%edx\n", (unsigned int)((cci->class.index * sizeof(kuint)) + sizeof(Class) + classdata->static_field_length));
	fprintf(file, "\tmovl\t0x%08X(%%edx), %%ecx\n", (unsigned int)offsetof(Class, classdata));

	fprintf(file, "\tmovl\t0x%08X(%%esp), %%ecx\n", (unsigned int)(object_offset * sizeof(kuint)));
	fprintf(file, "\tmovl\t0x%08X(%%ecx), %%edx\n", (unsigned int)offsetof(Object, class));
	fprintf(file, "\tmovl\t0x%08X(%%edx), %%ecx\n", (unsigned int)offsetof(Class, field));
	fprintf(file, "\tmovl\t0x%08X(%%eax), %%edx\n", (unsigned int)((fci->ref.index * sizeof(kuint)) + sizeof(Class) + classdata->static_field_length));

	fprintf(file, "\tmovl\t(%%ecx, %%edx, 4), %%edx\n");

	switch(opcode) {
		case OPCODE_getfield:
			fprintf(file, "\tpopl\t%%eax\n");
			fprintf(file, "\tpushl\t0x%08X(%%eax, %%edx, 1)\n", (unsigned int)offsetof(Object, data));
			break;
		case OPCODE_putfield:
			fprintf(file, "\tmovl\t0x%08X(%%esp), %%eax\n", (unsigned int)(object_offset * sizeof(kuint)));
			fprintf(file, "\tpopl\t0x%08X(%%eax, %%edx, 1)\n", (unsigned int)offsetof(Object, data));
			fprintf(file, "\tpopl\t%%eax\n");
			break;
		default:
			abort();
	}
}


void opcode_invokex(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);
	ConstantInfo* mci = constant_resolve(env, classinfo, read_u2(code, code_index), 1);
	ConstantInfo* cci = constant_resolve(env, classinfo, mci->ref.class_index, 1);
	kuint method_flags = (opcode == OPCODE_invokestatic) ? (FLAG_STATIC) : (0);

	parse_method_descriptor(mci->ref.descriptor_handle, &method_flags);

	switch(opcode) {
		case OPCODE_invokevirtual:
			// FIXME RF 2000-09-09.  There must be a better way to do this.
			fprintf(file, "\tmovl\t0x%08X(%%ebp), %%eax\n", (unsigned int)class_offset);
			fprintf(file, "\tmovl\t0x%08X(%%eax), %%edx\n", (unsigned int)((cci->class.index * sizeof(kuint)) + sizeof(Class) + classdata->static_field_length));
			fprintf(file, "\tmovl\t0x%08X(%%edx), %%ecx\n", (unsigned int)offsetof(Class, classdata));

			fprintf(file, "\tmovl\t0x%08X(%%esp), %%eax\n", (unsigned int)(((method_flags & FLAG_SIZE_MASK) - 1) * sizeof(kuint)));
			fprintf(file, "\tmovl\t0x%08X(%%eax), %%edx\n", (unsigned int)offsetof(Object, class));
			break;
		case OPCODE_invokespecial:
		case OPCODE_invokestatic:
			fprintf(file, "\tmovl\t0x%08X(%%ebp), %%eax\n", (unsigned int)class_offset);
			fprintf(file, "\tmovl\t0x%08X(%%eax), %%edx\n", (unsigned int)((cci->class.index * sizeof(kuint)) + sizeof(Class) + classdata->static_field_length));

			break;
		default:
			abort();
	}

	fprintf(file, "\tmovl\t0x%08X(%%edx), %%ecx\n", (unsigned int)offsetof(Class, method));
	fprintf(file, "\tpushl\t%%edx\n");
	fprintf(file, "\tmovl\t0x%08X(%%eax), %%edx\n", (unsigned int)((mci->ref.index * sizeof(kuint)) + sizeof(Class) + classdata->static_field_length));
	fprintf(file, "\tmovl\t(%%ecx, %%edx, 4), %%eax\n");

	fprintf(file, "\tpushl\t%%eax\n");
	fprintf(file, "\tpushl\t0x%08X(%%ebp)\n", (unsigned int)environment_offset);
	fprintf(file, "\tcall\t*%%eax\n");

	fprintf(file, "\taddl\t$0x%08X, %%esp\n", (unsigned int)(((method_flags & FLAG_SIZE_MASK) + 3) * sizeof(kuint)));

	if(method_flags & (FLAG_RETURN_REFERENCE | FLAG_RETURN_VALUE32 | FLAG_RETURN_VALUE64)) {
		fprintf(file, "\tpushl\t%%eax\n");
		if(method_flags & FLAG_RETURN_VALUE64) {
			fprintf(file, "\tpushl\t%%edx\n");
		}
	}
}

void opcode_new(Environment* env, ClassInfo* classinfo, ClassData* classdata, MethodData* methoddata, kuint8* code, kuint* code_index, FILE* file) {
	kuint opcode = read_u1(code, code_index);
	ConstantInfo* cci = constant_resolve(env, classinfo, read_u2(code, code_index), 1);

	if(opcode == OPCODE_new) {
		fprintf(file, "\tmovl\t0x%08X(%%ebp), %%eax\n", (unsigned int)class_offset);
		fprintf(file, "\tmovl\t0x%08X(%%eax), %%edx\n", (unsigned int)((cci->class.index * sizeof(kuint)) + sizeof(Class) + classdata->static_field_length));
		fprintf(file, "\tmovl\t0x%08X(%%edx), %%ecx\n", (unsigned int)offsetof(Class, classdata)); // Make sure the class is loaded
		fprintf(file, "\tpushl\t%%edx\n");
		fprintf(file, "\tpushl\t0x%08X(%%ebp)\n", (unsigned int)environment_offset);
		fprintf(file, "\tmovl\t$0x%08X, %%eax\n", (unsigned int)object_create);
		fprintf(file, "\tcall\t*%%eax /* call object_create */\n");
		fprintf(file, "\taddl\t$8, %%esp\n");
		fprintf(file, "\tpushl\t%%eax\n");
	} else {
		abort();
	}
}

#define OPCODE(o,h) {#o, h,}

struct opcode opcode_table[256] = {
	OPCODE(nop, opcode_nop),
	OPCODE(aconst_null, opcode_xpush),
	OPCODE(iconst_m1, opcode_xpush),
	OPCODE(iconst_0, opcode_xpush),
	OPCODE(iconst_1, opcode_xpush),
	OPCODE(iconst_2, opcode_xpush),
	OPCODE(iconst_3, opcode_xpush),
	OPCODE(iconst_4, opcode_xpush),
	OPCODE(iconst_5, opcode_xpush),
	OPCODE(lconst_0, knull),
	OPCODE(lconst_1, knull),
	OPCODE(fconst_0, opcode_xpush),
	OPCODE(fconst_1, opcode_xpush),
	OPCODE(fconst_2, opcode_xpush),
	OPCODE(dconst_0, knull),
	OPCODE(dconst_1, knull),
	OPCODE(bipush, opcode_xpush),
	OPCODE(sipush, opcode_xpush),
	OPCODE(ldc, opcode_xpush),
	OPCODE(ldc_w, opcode_xpush),
	OPCODE(ldc2_w, knull),
	OPCODE(iload, opcode_xload),
	OPCODE(lload, knull),
	OPCODE(fload, knull),
	OPCODE(dload, knull),
	OPCODE(aload, opcode_xload),
	OPCODE(iload_0, opcode_xload),
	OPCODE(iload_1, opcode_xload),
	OPCODE(iload_2, opcode_xload),
	OPCODE(iload_3, opcode_xload),
	OPCODE(lload_0, knull),
	OPCODE(lload_1, knull),
	OPCODE(lload_2, knull),
	OPCODE(lload_3, knull),
	OPCODE(fload_0, knull),
	OPCODE(fload_1, knull),
	OPCODE(fload_2, knull),
	OPCODE(fload_3, knull),
	OPCODE(dload_0, knull),
	OPCODE(dload_1, knull),
	OPCODE(dload_2, knull),
	OPCODE(dload_3, knull),
	OPCODE(aload_0, opcode_xload),
	OPCODE(aload_1, opcode_xload),
	OPCODE(aload_2, opcode_xload),
	OPCODE(aload_3, opcode_xload),
	OPCODE(iaload, knull),
	OPCODE(laload, knull),
	OPCODE(faload, knull),
	OPCODE(daload, knull),
	OPCODE(aaload, knull),
	OPCODE(baload, knull),
	OPCODE(caload, knull),
	OPCODE(saload, knull),
	OPCODE(istore, opcode_xstore),
	OPCODE(lstore, knull),
	OPCODE(fstore, knull),
	OPCODE(dstore, knull),
	OPCODE(astore, opcode_xstore),
	OPCODE(istore_0, opcode_xstore),
	OPCODE(istore_1, opcode_xstore),
	OPCODE(istore_2, opcode_xstore),
	OPCODE(istore_3, opcode_xstore),
	OPCODE(lstore_0, knull),
	OPCODE(lstore_1, knull),
	OPCODE(lstore_2, knull),
	OPCODE(lstore_3, knull),
	OPCODE(fstore_0, knull),
	OPCODE(fstore_1, knull),
	OPCODE(fstore_2, knull),
	OPCODE(fstore_3, knull),
	OPCODE(dstore_0, knull),
	OPCODE(dstore_1, knull),
	OPCODE(dstore_2, knull),
	OPCODE(dstore_3, knull),
	OPCODE(astore_0, opcode_xstore),
	OPCODE(astore_1, opcode_xstore),
	OPCODE(astore_2, opcode_xstore),
	OPCODE(astore_3, opcode_xstore),
	OPCODE(iastore, knull),
	OPCODE(lastore, knull),
	OPCODE(fastore, knull),
	OPCODE(dastore, knull),
	OPCODE(aastore, knull),
	OPCODE(bastore, knull),
	OPCODE(castore, knull),
	OPCODE(sastore, knull),
	OPCODE(pop, opcode_stack),
	OPCODE(pop2, opcode_stack),
	OPCODE(dup, opcode_stack),
	OPCODE(dup_x1, knull),
	OPCODE(dup_x2, knull),
	OPCODE(dup2, knull),
	OPCODE(dup2_x1, knull),
	OPCODE(dup2_x2, knull),
	OPCODE(swap, knull),
	OPCODE(iadd, opcode_iop),
	OPCODE(ladd, knull),
	OPCODE(fadd, knull),
	OPCODE(dadd, knull),
	OPCODE(isub, opcode_iop),
	OPCODE(lsub, knull),
	OPCODE(fsub, knull),
	OPCODE(dsub, knull),
	OPCODE(imul, opcode_iop),
	OPCODE(lmul, knull),
	OPCODE(fmul, knull),
	OPCODE(dmul, knull),
	OPCODE(idiv, opcode_iop),
	OPCODE(ldiv, knull),
	OPCODE(fdiv, knull),
	OPCODE(ddiv, knull),
	OPCODE(irem, opcode_iop),
	OPCODE(lrem, knull),
	OPCODE(frem, knull),
	OPCODE(drem, knull),
	OPCODE(ineg, opcode_iop),
	OPCODE(lneg, knull),
	OPCODE(fneg, knull),
	OPCODE(dneg, knull),
	OPCODE(ishl, opcode_iop),
	OPCODE(lshl, knull),
	OPCODE(ishr, opcode_iop),
	OPCODE(lshr, knull),
	OPCODE(iushr, opcode_iop),
	OPCODE(lushr, knull),
	OPCODE(iand, opcode_iop),
	OPCODE(land, knull),
	OPCODE(ior, opcode_iop),
	OPCODE(lor, knull),
	OPCODE(ixor, opcode_iop),
	OPCODE(lxor, knull),
	OPCODE(iinc, opcode_iinc),
	OPCODE(i2l, knull),
	OPCODE(i2f, knull),
	OPCODE(i2d, knull),
	OPCODE(l2i, knull),
	OPCODE(l2f, knull),
	OPCODE(l2d, knull),
	OPCODE(f2i, knull),
	OPCODE(f2l, knull),
	OPCODE(f2d, knull),
	OPCODE(d2i, knull),
	OPCODE(d2l, knull),
	OPCODE(d2f, knull),
	OPCODE(i2b, knull),
	OPCODE(i2c, knull),
	OPCODE(i2s, knull),
	OPCODE(lcmp, knull),
	OPCODE(fcmpl, knull),
	OPCODE(fcmpg, knull),
	OPCODE(dcmpl, knull),
	OPCODE(dcmpg, knull),
	OPCODE(ifeq, opcode_ifx),
	OPCODE(ifne, opcode_ifx),
	OPCODE(iflt, opcode_ifx),
	OPCODE(ifge, opcode_ifx),
	OPCODE(ifgt, opcode_ifx),
	OPCODE(ifle, opcode_ifx),
	OPCODE(if_icmpeq, opcode_if_xcmpx),
	OPCODE(if_icmpne, opcode_if_xcmpx),
	OPCODE(if_icmplt, opcode_if_xcmpx),
	OPCODE(if_icmpge, opcode_if_xcmpx),
	OPCODE(if_icmpgt, opcode_if_xcmpx),
	OPCODE(if_icmple, opcode_if_xcmpx),
	OPCODE(if_acmpeq, opcode_if_xcmpx),
	OPCODE(if_acmpne, opcode_if_xcmpx),
	OPCODE(goto, opcode_goto),
	OPCODE(jsr, knull),
	OPCODE(ret, knull),
	OPCODE(tableswitch, knull),
	OPCODE(lookupswitch, knull),
	OPCODE(ireturn, opcode_xreturn),
	OPCODE(lreturn, knull),
	OPCODE(freturn, knull),
	OPCODE(dreturn, knull),
	OPCODE(areturn, opcode_xreturn),
	OPCODE(return, opcode_xreturn),
	OPCODE(getstatic, knull),
	OPCODE(putstatic, knull),
	OPCODE(getfield, opcode_xfield),
	OPCODE(putfield, opcode_xfield),
	OPCODE(invokevirtual, opcode_invokex),
	OPCODE(invokespecial, opcode_invokex),
	OPCODE(invokestatic, opcode_invokex),
	OPCODE(invokeinterface, knull),
	OPCODE(xxxunusedxxx, knull),
	OPCODE(new, opcode_new),
	OPCODE(newarray, knull),
	OPCODE(anewarray, knull),
	OPCODE(arraylength, knull),
	OPCODE(athrow, knull),
	OPCODE(checkcast, knull),
	OPCODE(instanceof, knull),
	OPCODE(monitorenter, knull),
	OPCODE(monitorexit, knull),
	OPCODE(wide, knull),
	OPCODE(multianewarray, knull),
	OPCODE(ifnull, opcode_ifx),
	OPCODE(ifnonnull, opcode_ifx),
	OPCODE(goto_w, knull),
	OPCODE(jsr_w, knull),
	OPCODE(breakpoint, knull),
};

