/*
 * Copyright (C) 2000 Robert Fitzsimons
 */

#include "types.h"
#include "environment.h"
#include "class.h"
#include "classcode.h"
#include "classdata.h"
#include "classinfo.h"
#include "unicode.h"
#include "util.h"
#include <dmalloc.h>

kuint code_handle;

void init_classdata() {
	code_handle = unicode_put_ascii("Code");
}

ClassData* classdata_build(Environment* env, ClassInfo* classinfo);
ClassData* classdata_build_array(Environment* env, ClassInfo** classinfo);

void classdata_build_methoddata(Environment* env, ClassInfo* classinfo, ClassData* classdata);
void classdata_build_fielddata(Environment* env, ClassInfo* classinfo, ClassData* classdata);
void classdata_build_interfacedata(Environment* env, ClassInfo* classinfo, ClassData* classdata);

void parse_methodinfo_code(Environment* env, ClassInfo* classinfo, MethodInfo* methodinfo, MethodData* methoddata);

kuint read_classfile(Unicode* unicode, kuint8** buffer, kuint* buffer_length) {
	kuint8* classname = knull;
	kuint8* path = "classes/";
	kuint8* ext = ".class";
	kuint8* utf8 = knull;
	kuint utf8_length = 0;
	kuint rc = 0;

	utf8 = unicode_to_utf8(unicode, &utf8_length);
	classname = memory_allocate(utf8_length + strlen(path) + strlen(ext) + 1, sizeof(kuint8));
	memory_copy(&classname[0], path, strlen(path));
	memory_copy(&classname[strlen(path)], utf8, utf8_length);
	memory_copy(&classname[strlen(path) + utf8_length], ext, strlen(ext));

	rc = read_file(classname, buffer, buffer_length);

	memory_free(classname);
	memory_free(utf8);

	return rc;
}

ClassData* classdata_find(Environment* env, kuint class_handle) {
	ClassData* classdata = knull;
	ClassInfo* classinfo = knull;
	Unicode* classname = knull;
	kuint8* buffer = knull;
	kuint buffer_length = 0;
	kuint index = 0;

	classdata = classdata_lookup(env, class_handle);

	if(classdata != knull) {
		return classdata;
	}

	classname = unicode_get(class_handle);

	if(classname->array[0] != '[') {
		if(read_classfile(classname, &buffer, &buffer_length) == 0) {
			classinfo = classinfo_build(buffer, buffer_length);
			classdata = classdata_build(env, classinfo);
		}
	} else {
		if(env->array_classdata == knull) {
			env->array_classdata = classdata_build_array(env, &env->array_classinfo);
		}
		classdata = env->array_classdata;
		classinfo = env->array_classinfo;

		if(classname->array[1] == 'L') {
			kuint handle = unicode_put_u2_array(&classname->array[2], (classname->length - 2));
			class_find(env, handle);
		} else {
			kuint handle = unicode_put_u2_array(&classname->array[1], (classname->length - 1));
			if(classname->array[1] == '[') {
				class_find(env, handle);
			}
		}
	}

	if(classdata == knull) {
		return knull;
	}

	if(classdata->method_data_count > 0) {
		classdata->method = memory_allocate(classdata->method_data_count, sizeof(kuint));
		for(index = 0; index < classdata->method_data_count; index++) {
			MethodData* method_data = classdata->method_data[index];
			if(!(method_data->flags & (FLAG_RESOLVED))) {
				if(!(method_data->flags & (FLAG_NATIVE | FLAG_ABSTRACT))) {
					method_data->flags |= FLAG_RESOLVED;
					bytecode_to_nativecode(env, classinfo, classdata, method_data);
				} else {
					method_data->flags |= FLAG_RESOLVED;
					method_data->code = (void*)0xFFF10000;
				}
			}
			classdata->method[index] = (kuint)method_data->code;
		}
	}

	if(classdata->field_data_count > 0) {
		classdata->field = memory_allocate(classdata->field_data_count, sizeof(kuint));
		for(index = 0; index < classdata->field_data_count; index++) {
			classdata->field[index] = classdata->field_data[index]->offset;
		}
	}

	if(classinfo->constant_index > 0) {
		classdata->constant_data_count = classinfo->constant_index;
		classdata->constant_data = memory_allocate(classdata->constant_data_count, sizeof(ConstantData));

		for(index = 0; index < classinfo->constant_count; index++) {
			ConstantInfo* ci = &classinfo->constant[index];
			kuint i = ci->info.index;
			if(ci->info.tag & CONSTANT_Keep) {
				switch(ci->info.tag & CONSTANT_Mask) {
					case CONSTANT_Class:
						classdata->constant_data[i].class.tag = (ci->info.tag & CONSTANT_Mask);
						classdata->constant_data[i].class.class_handle = ci->class.class_handle;
						break;
					case CONSTANT_MethodReference:
					case CONSTANT_FieldReference:
						classdata->constant_data[i].ref.tag = (ci->info.tag & CONSTANT_Mask);
						classdata->constant_data[i].ref.class_index = classinfo->constant[ci->ref.class_index].class.index;
						classdata->constant_data[i].ref.name_handle = ci->ref.name_handle;
						classdata->constant_data[i].ref.descriptor_handle = ci->ref.descriptor_handle;
						break;
				}
			}
		}
	}

	classdata_print(classdata);

	if(classinfo) {
		classinfo_free(classinfo);
	}
	if(buffer) {
		memory_free(buffer);
	}

	return classdata;
}

ClassData* classdata_lookup(Environment* env, kuint class_handle) {
	ClassData* classdata = knull;

	classdata = hashmap_get(env->classdata_map, class_handle);

	return classdata;
}

ConstantInfo* constant_resolve(Environment* env, ClassInfo* classinfo, kuint index, kuint keep) {
	ConstantInfo* ci = &classinfo->constant[index];

	if(!(ci->info.tag & CONSTANT_Resolved)) {
		ConstantInfo cci = classinfo->constant[index];
		switch(ci->info.tag & CONSTANT_Mask) {
			case CONSTANT_Info_Utf8:
				ci->unicode.tag = CONSTANT_Resolved | CONSTANT_Unicode;
				ci->unicode.hash = unicode_put_utf8(cci.utf8_info.bytes, cci.utf8_info.length);
				break;
			case CONSTANT_Info_Class:
				ci->class.tag = CONSTANT_Resolved | CONSTANT_Class;
				ci->class.class_handle = (constant_resolve(env, classinfo, cci.class_info.name_index, 0))->unicode.hash;
				break;
			case CONSTANT_Info_FieldRef:
			case CONSTANT_Info_MethodRef:
			case CONSTANT_Info_InterfaceMethodRef:
				if((ci->info.tag & CONSTANT_Mask) != CONSTANT_Info_FieldRef) {
					ci->ref.tag = CONSTANT_Resolved | CONSTANT_MethodReference;
				} else {
					ci->ref.tag = CONSTANT_Resolved | CONSTANT_FieldReference;
				}
				ci->ref.class_index = cci.ref_info.class_index;
				constant_resolve(env, classinfo, ci->ref.class_index, keep);
				{
					ConstantInfo* nci = constant_resolve(env, classinfo, cci.ref_info.nametype_index, 0);
					ci->ref.name_handle = nci->nametype.name_handle;
					ci->ref.descriptor_handle = nci->nametype.descriptor_handle;
				}
				break;
			case CONSTANT_Info_NameAndType:
				ci->nametype.tag = CONSTANT_Resolved | CONSTANT_NameAndType;
				ci->nametype.name_handle = (constant_resolve(env, classinfo, cci.nametype_info.name_index, 0))->unicode.hash;
				ci->nametype.descriptor_handle = (constant_resolve(env, classinfo, cci.nametype_info.descriptor_index, 0))->unicode.hash;
				break;
			case CONSTANT_Info_String:
				print_ascii("*** FIXME String ***");
				abort();
		}
	}

	if(keep && (!(ci->info.tag & CONSTANT_Keep))) {
		ci->info.tag |= CONSTANT_Keep;
		ci->info.index = classinfo->constant_index++;
	}

	return ci;
}

void classdata_build_methoddata(Environment* env, ClassInfo* classinfo, ClassData* classdata) {
	if(classdata->superclass != knull) {
		kuint index = 0;
		kuint index2 = 0;
		kuint count = 0;
		kuint found = 0;

		for(index = 0; index < classinfo->method_count; index++) {
			MethodInfo* mi = &classinfo->method[index];

			mi->access_flags <<= 16;
			mi->name_index = (constant_resolve(env, classinfo, mi->name_index, 0))->unicode.hash;
			mi->descriptor_index = (constant_resolve(env, classinfo, mi->descriptor_index, 0))->unicode.hash;

			for(index2 = 0; index2 < classdata->superclass->method_data_count; index2++) {
				MethodData* smd = classdata->superclass->method_data[index2];
				if((mi->name_index == smd->name_handle) && (mi->descriptor_index == smd->descriptor_handle)) {
					found++;
					break;
				}
			}
			if(index2 >= classdata->superclass->method_data_count) {
				count++;
			}
		}

		if(count || found) {
			classdata->method_data_count = classdata->superclass->method_data_count + count;
			classdata->method_data = memory_allocate(classdata->method_data_count, sizeof(MethodData*));
			memory_copy(classdata->method_data, classdata->superclass->method_data, classdata->superclass->method_data_count * sizeof(MethodData*));
			count = classdata->superclass->method_data_count;
		} else {
			classdata->method_data_count = classdata->superclass->method_data_count;
			classdata->method_data = classdata->superclass->method_data;
		}

		for(index = 0; index < classinfo->method_count; index++) {
			MethodInfo* mi = &classinfo->method[index];
			MethodData* md = memory_allocate(1, sizeof(MethodData));

			md->classdata = classdata;
			md->flags = mi->access_flags;
			md->name_handle = mi->name_index;
			md->descriptor_handle = mi->descriptor_index;
			parse_method_descriptor(md->descriptor_handle, &md->flags);
			parse_methodinfo_code(env, classinfo, mi, md);

			for(index2 = 0; index2 < classdata->superclass->method_data_count; index2++) {
				MethodData* smd = classdata->superclass->method_data[index2];
				if((mi->name_index == smd->name_handle) && (mi->descriptor_index == smd->descriptor_handle)) {
					classdata->method_data[index2] = md;
					break;
				}
			}
			if(index2 >= classdata->superclass->method_data_count) {
				classdata->method_data[count] = md;
				count++;
			}
		}
	} else {
		kuint index = 0;

		if(classinfo->method_count > 0) {
			classdata->method_data_count = classinfo->method_count;
			classdata->method_data = memory_allocate(classdata->method_data_count, sizeof(MethodData*));
		}

		for(index = 0; index < classinfo->method_count; index++) {
			MethodInfo* mi = &classinfo->method[index];
			MethodData* md = memory_allocate(1, sizeof(MethodData));

			md->classdata = classdata;
			md->flags = mi->access_flags << 16;
			md->name_handle = (constant_resolve(env, classinfo, mi->name_index, 0))->unicode.hash;
			md->descriptor_handle = (constant_resolve(env, classinfo, mi->descriptor_index, 0))->unicode.hash;
			parse_method_descriptor(md->descriptor_handle, &md->flags);
			parse_methodinfo_code(env, classinfo, mi, md);

			classdata->method_data[index] = md;
		}
	}
}

void classdata_build_fielddata(Environment* env, ClassInfo* classinfo, ClassData* classdata) {
	kuint index = 0;
	kuint count = 0;

	if(classdata->superclass != knull) {
		if(classinfo->field_count) {
			classdata->instance_field_length = classdata->superclass->instance_field_length;
			classdata->static_field_length = classdata->superclass->static_field_length;
			classdata->field_data_count = classdata->superclass->field_data_count + classinfo->field_count;
			classdata->field_data = memory_allocate(classdata->field_data_count, sizeof(FieldData*));
			memory_copy(classdata->field_data, classdata->superclass->field_data, classdata->superclass->field_data_count * sizeof(FieldData*));
			count = classdata->superclass->field_data_count;
		} else {
			classdata->instance_field_length = classdata->superclass->instance_field_length;
			classdata->static_field_length = classdata->superclass->static_field_length;
			classdata->field_data_count = classdata->superclass->field_data_count;
			classdata->field_data = classdata->superclass->field_data;
		}
	} else {
		if(classinfo->field_count) {
			classdata->field_data_count = classinfo->field_count;
			classdata->field_data = memory_allocate(classdata->field_data_count, sizeof(FieldData*));
		}
	}

	for(index = 0; index < classinfo->field_count; index++) {
		FieldInfo* fi = &classinfo->field[index];
		FieldData* fd = memory_allocate(1, sizeof(FieldData));

		fd->flags = fi->access_flags << 16;
		fd->name_handle = (constant_resolve(env, classinfo, fi->name_index, 0))->unicode.hash;
		fd->descriptor_handle = (constant_resolve(env, classinfo, fi->descriptor_index, 0))->unicode.hash;

		parse_field_descriptor(fd->descriptor_handle, &fd->flags);

		if(!(fi->access_flags & FLAG_STATIC)) {
			fd->offset = classdata->instance_field_length;
			classdata->instance_field_length += ((fd->flags & FLAG_SIZE_MASK) == 8) ? 8 : 4;
		} else {
			fd->offset = classdata->static_field_length;
			classdata->static_field_length += ((fd->flags & FLAG_SIZE_MASK) == 8) ? 8 : 4;
		}

		classdata->field_data[count++] = fd;
	}
}

void classdata_build_interfacedata(Environment* env, ClassInfo* classinfo, ClassData* classdata) {
	if(classdata->superclass != knull) {
		if(classinfo->interface_count > 0) {
			kuint index = 0;

			classdata->interface_data_count = classinfo->interface_count;
			classdata->interface_data = memory_allocate(classinfo->interface_count, sizeof(InterfaceData*));

			for(index = 0; index < classinfo->interface_count; index++) {
				kuint ii = (constant_resolve(env, classinfo, classinfo->interface[index], 0))->class.class_handle;
				classdata->interface_data[index] = memory_allocate(1, sizeof(InterfaceData));
				classdata->interface_data[index]->classdata = class_find(env, ii)->classdata;
			}
		}
	}
}

ClassData* classdata_build(Environment* env, ClassInfo* classinfo) {
	ClassData* classdata = knull;

	classdata = memory_allocate(1, sizeof(ClassData));

	classdata->name_handle = (constant_resolve(env, classinfo, classinfo->thisclass, 0))->class.class_handle;
	hashmap_put(env->classdata_map, classdata->name_handle, classdata);

	classdata->flags = classinfo->access_flags << 16;

	if(classinfo->superclass != 0) {
		kuint class_handle = (constant_resolve(env, classinfo, classinfo->superclass, 0))->class.class_handle;
		classdata->superclass = class_find(env, class_handle)->classdata;
	}

	classdata_build_methoddata(env, classinfo, classdata);
	classdata_build_fielddata(env, classinfo, classdata);
	classdata_build_interfacedata(env, classinfo, classdata);

	return classdata;
}

kuint find_methoddata_index(ClassData* classdata, kuint name_handle, kuint descriptor_handle) {
	kuint index = 0;

	for(index = 0; index < classdata->method_data_count; index++) {
		if((name_handle == classdata->method_data[index]->name_handle) && (descriptor_handle == classdata->method_data[index]->descriptor_handle)) {
			return index;
		}
	}

	return -1;
}

kuint find_fielddata_index(ClassData* classdata, kuint name_handle, kuint descriptor_handle) {
	kuint index = 0;

	for(index = 0; index < classdata->field_data_count; index++) {
		if((name_handle == classdata->field_data[index]->name_handle) && (descriptor_handle == classdata->field_data[index]->descriptor_handle)) {
			return index;
		}
	}

	return -1;
}

void classdata_print(ClassData* classdata) {
	kuint index = 0;

	print_ascii("0x");
	print_u4((kuint)classdata);
	print_nl();

	print_u4(classdata->name_handle);
	print_sp();
	print_unicode(classdata->name_handle);
	print_nl();

	if(classdata->superclass != knull) {
		print_u4(classdata->superclass->name_handle);
		print_sp();
		print_unicode(classdata->superclass->name_handle);
	} else {
		print_u4(0);
	}
	print_nl();

	print_u4(classdata->flags);
	print_nl();

	print_u4(classdata->method_data_count);
	print_nl();
	for(index = 0; index < classdata->method_data_count; index++) {
		print_sp();
		print_u4(index);
		print_sp();
		classdata_print_methoddata(classdata->method_data[index]);
		print_nl();
	}

	print_u4(classdata->field_data_count);
	print_sp();
	print_u4(classdata->instance_field_length);
	print_sp();
	print_u4(classdata->static_field_length);
	print_nl();
	for(index = 0; index < classdata->field_data_count; index++) {
		print_sp();
		print_u4(index);
		print_sp();
		print_ascii("0x");
		print_u4((kuint)classdata->field_data[index]);
		print_sp();
		print_u4(classdata->field_data[index]->name_handle);
		print_sp();
		print_u4(classdata->field_data[index]->descriptor_handle);
		print_sp();
		print_u4(classdata->field_data[index]->flags);
		print_sp();
		print_u4(classdata->field_data[index]->offset);
		print_nl();
	}

	print_u4(classdata->interface_data_count);
	print_nl();
	for(index = 0; index < classdata->interface_data_count; index++) {
		print_sp();
		print_u4(index);
		print_sp();
		print_u4(classdata->interface_data[index]->classdata->name_handle);
		print_nl();
	}

	print_u4(classdata->constant_data_count);
	print_nl();
	for(index = 0; index < classdata->constant_data_count; index++) {
		print_sp();
		print_u4(index);
		print_sp();
		print_u4(classdata->constant_data[index].ref.tag);
		print_sp();
		print_u4(classdata->constant_data[index].ref.class_index);
		print_sp();
		print_u4(classdata->constant_data[index].ref.name_handle);
		print_sp();
		print_u4(classdata->constant_data[index].ref.descriptor_handle);
		print_nl();
	}
}

void classdata_print_methoddata(MethodData* methoddata) {
	print_ascii("0x");
	print_u4((kuint)methoddata);
	print_sp();
	print_u4(methoddata->name_handle);
	print_sp();
	print_u4(methoddata->descriptor_handle);
	print_sp();
	print_u4(methoddata->flags);
	print_sp();
	print_u2(methoddata->max_stack);
	print_sp();
	print_u2(methoddata->max_locals);
	print_ascii("  0x");
	print_u4((kuint)methoddata->code);
}

#define constant_unicode(i,h) ({ \
	(*classinfo)->constant[(i)].unicode.tag = CONSTANT_Unicode; \
	(*classinfo)->constant[(i)].unicode.hash = (h); \
});
#define constant_class(i,n) ({ \
	(*classinfo)->constant[(i)].class_info.tag = CONSTANT_Info_Class; \
	(*classinfo)->constant[(i)].class_info.name_index = (n); \
});
#define constant_nametype(i,n,t) ({ \
	(*classinfo)->constant[(i)].nametype_info.tag = CONSTANT_Info_NameAndType; \
	(*classinfo)->constant[(i)].nametype_info.name_index = (n); \
	(*classinfo)->constant[(i)].nametype_info.descriptor_index = (t); \
});
#define field(i,n,d,f) ({ \
	(*classinfo)->field[(i)].access_flags = (f); \
	(*classinfo)->field[(i)].name_index = (n); \
	(*classinfo)->field[(i)].descriptor_index = (d); \
});
#define method(i,n,d,f) ({ \
	(*classinfo)->method[(i)].access_flags = (f); \
	(*classinfo)->method[(i)].name_index = (n); \
	(*classinfo)->method[(i)].descriptor_index = (d); \
});

ClassData* classdata_build_array(Environment* env, ClassInfo** classinfo) {
	ClassData* classdata = knull;

	(*classinfo) = memory_allocate(1, sizeof(ClassInfo));

	(*classinfo)->constant_count = 10;
	(*classinfo)->constant = memory_allocate((*classinfo)->constant_count, sizeof(ConstantInfo));
	constant_class(1, 5);
	constant_class(2, 6);
	constant_class(3, 7);
	constant_class(4, 8);
	constant_unicode(5, unicode_put_ascii("[<array>"));
	constant_unicode(6, unicode_put_ascii("java/lang/Object"));
	constant_unicode(7, unicode_put_ascii("java/lang/Cloneable"));
	constant_unicode(8, unicode_put_ascii("java/io/Serializable"));

	(*classinfo)->interface_count = 2;
	(*classinfo)->interface = memory_allocate((*classinfo)->interface_count, sizeof(kuint16));
	(*classinfo)->interface[0] = 3;
	(*classinfo)->interface[1] = 4;

	(*classinfo)->access_flags = ((FLAG_PUBLIC | FLAG_FINAL| FLAG_SUPER) >> 16);
	(*classinfo)->thisclass = 1;
	(*classinfo)->superclass = 2;

	classdata = classdata_build(env, (*classinfo));

	classdata->flags |= FLAG_ARRAY;

	return classdata;
}

void parse_method_descriptor(kuint descriptor_handle, kuint* flags) {
	Unicode* descriptor = unicode_get(descriptor_handle);
	kuint c = 0;
	kuint index = 0;
	kuint16* array = descriptor->array;
	kuint count = 0;
	kuint return_type = 0;

	for(index = 1; (c = array[index]) != ')'; index++) {
		switch(c) {
			case '[':
				while((c = array[++index]) == '[') {
				}
				if(c != 'L') {
					count++;
					break;
				}
			case 'L':
				while((c = array[++index]) != ';') {
				}
				count++;
				break;
			case 'D':
			case 'J':
				count++;
			default:
				count++;
				break;
		}
	}

	switch(c = array[++index]) {
		case '[':
		case 'L':
			return_type = FLAG_RETURN_REFERENCE;
			break;
		case 'D':
		case 'J':
			return_type = FLAG_RETURN_VALUE64;
			break;
		case 'V':
			return_type = FLAG_RETURN_VOID;
			break;
		default:
			return_type = FLAG_RETURN_VALUE32;
			break;
	}

	if(!((*flags) & FLAG_STATIC)) {
		count++;
	}

	(*flags) |= count | return_type;
}

void parse_field_descriptor(kuint descriptor_handle, kuint* flags) {
	Unicode* descriptor = unicode_get(descriptor_handle);
	kuint size = 0;

	switch(descriptor->array[0]) {
		case '[':
		case 'L':
			size = sizeof(void*);
			break;
		case 'Z':
		case 'B':
			size = sizeof(kuint8);
			break;
		case 'C':
		case 'S':
			size = sizeof(kuint16);
			break;
		case 'F':
		case 'I':
			size = sizeof(kuint32);
			break;
		case 'D':
		case 'J':
			size = sizeof(kuint64);
			break;
	}

	(*flags) |= size;
}

void parse_methodinfo_code(Environment* env, ClassInfo* classinfo, MethodInfo* methodinfo, MethodData* methoddata) {
	kuint index = 0;

	for(index = 0; index < methodinfo->attribute_count; index++) {
		if((constant_resolve(env, classinfo, methodinfo->attribute[index].name_index, 0))->unicode.hash == code_handle) {
			kuint8* buffer = methodinfo->attribute[index].array;
			kuint buffer_index = 0;
			methoddata->max_stack = read_u2(buffer, &buffer_index);
			methoddata->max_locals = read_u2(buffer, &buffer_index);
			methoddata->code_length = read_u4(buffer, &buffer_index);
			methoddata->code = (void*)&buffer[buffer_index];
		}
	}
}

