/*
 * Copyright (C) 2000 Robert Fitzsimons
 */

#include "types.h"
#include "util.h"
#include "unicode.h"
#include "classinfo.h"

void init_classinfo() {
}

void classinfo_parse_internal(ClassInfo* classinfo, kuint8* buffer, kuint* buffer_index);
void classinfo_print_internal(ClassInfo* classinfo);

ClassInfo* classinfo_build(kuint8* buffer, kuint buffer_length) {
	ClassInfo* classinfo = knull;
	kuint buffer_index = 0;

	classinfo = memory_allocate(1, sizeof(ClassInfo));

	classinfo_parse_internal(classinfo, buffer, &buffer_index);
//	classinfo_print_internal(classinfo);

	return classinfo;
}

void classinfo_parse_internal(ClassInfo* classinfo, kuint8* buffer, kuint* buffer_index) {
	kuint index = 0;
	kuint index2 = 0;

	classinfo->magic = read_u4(buffer, buffer_index);
	classinfo->minor_version = read_u2(buffer, buffer_index);
	classinfo->major_version = read_u2(buffer, buffer_index);

	classinfo->constant_count = read_u2(buffer, buffer_index);
	if(classinfo->constant_count > 0) {
		classinfo->constant = memory_allocate(classinfo->constant_count, sizeof(ConstantInfo));
		for(index = 1; index < classinfo->constant_count; index++) {
			classinfo->constant[index].info.tag = read_u1(buffer, buffer_index);
			switch(classinfo->constant[index].info.tag) {
				case CONSTANT_Info_Utf8:
					classinfo->constant[index].utf8_info.length = read_u2(buffer, buffer_index);
					classinfo->constant[index].utf8_info.bytes = &buffer[(*buffer_index)];
					(*buffer_index) += classinfo->constant[index].utf8_info.length;
					break;
				case CONSTANT_Info_Integer:
				case CONSTANT_Info_Float:
					classinfo->constant[index].v32_info.bytes = read_u4(buffer, buffer_index);
					break;
				case CONSTANT_Info_Long:
				case CONSTANT_Info_Double:
					classinfo->constant[index].v64_info.high_bytes = read_u4(buffer, buffer_index);
					classinfo->constant[index].v64_info.low_bytes = read_u4(buffer, buffer_index);
					index++;
					break;
				case CONSTANT_Info_String:
					classinfo->constant[index].string_info.string_index = read_u2(buffer, buffer_index);
					break;
				case CONSTANT_Info_Class:
					classinfo->constant[index].class_info.name_index = read_u2(buffer, buffer_index);
					break;
				case CONSTANT_Info_FieldRef:
				case CONSTANT_Info_MethodRef:
				case CONSTANT_Info_InterfaceMethodRef:
					classinfo->constant[index].ref_info.class_index = read_u2(buffer, buffer_index);
					classinfo->constant[index].ref_info.nametype_index = read_u2(buffer, buffer_index);
					break;
				case CONSTANT_Info_NameAndType:
					classinfo->constant[index].nametype_info.name_index = read_u2(buffer, buffer_index);
					classinfo->constant[index].nametype_info.descriptor_index = read_u2(buffer, buffer_index);
					break;
			}
		}
	}

	classinfo->access_flags = read_u2(buffer, buffer_index);
	classinfo->thisclass = read_u2(buffer, buffer_index);
	classinfo->superclass = read_u2(buffer, buffer_index);

	classinfo->interface_count = read_u2(buffer, buffer_index);
	if(classinfo->interface_count > 0) {
		classinfo->interface = memory_allocate(classinfo->interface_count, sizeof(kuint));
		for(index = 0; index < classinfo->interface_count; index++) {
			classinfo->interface[index] = read_u2(buffer, buffer_index);
		}
	}

	classinfo->field_count = read_u2(buffer, buffer_index);
	if(classinfo->field_count > 0) {
		classinfo->field = memory_allocate(classinfo->field_count, sizeof(FieldInfo));
		for(index = 0; index < classinfo->field_count; index++) {
			classinfo->field[index].access_flags = read_u2(buffer, buffer_index);
			classinfo->field[index].name_index = read_u2(buffer, buffer_index);
			classinfo->field[index].descriptor_index = read_u2(buffer, buffer_index);
			classinfo->field[index].attribute_count = read_u2(buffer, buffer_index);
			if(classinfo->field[index].attribute_count > 0) {
				classinfo->field[index].attribute = memory_allocate(classinfo->field[index].attribute_count, sizeof(AttributeInfo));
				for(index2 = 0; index2 < classinfo->field[index].attribute_count; index2++) {
					classinfo->field[index].attribute[index2].name_index = read_u2(buffer, buffer_index);
					classinfo->field[index].attribute[index2].length = read_u4(buffer, buffer_index);
					classinfo->field[index].attribute[index2].array = &buffer[(*buffer_index)];
					(*buffer_index) += classinfo->field[index].attribute[index2].length;
				}
			}
		}
	}

	classinfo->method_count = read_u2(buffer, buffer_index);
	if(classinfo->method_count > 0) {
		classinfo->method = memory_allocate(classinfo->method_count, sizeof(MethodInfo));
		for(index = 0; index < classinfo->method_count; index++) {
			classinfo->method[index].access_flags = read_u2(buffer, buffer_index);
			classinfo->method[index].name_index = read_u2(buffer, buffer_index);
			classinfo->method[index].descriptor_index = read_u2(buffer, buffer_index);
			classinfo->method[index].attribute_count = read_u2(buffer, buffer_index);
			if(classinfo->method[index].attribute_count > 0) {
				classinfo->method[index].attribute = memory_allocate(classinfo->method[index].attribute_count, sizeof(AttributeInfo));
				for(index2 = 0; index2 < classinfo->method[index].attribute_count; index2++) {
					classinfo->method[index].attribute[index2].name_index = read_u2(buffer, buffer_index);
					classinfo->method[index].attribute[index2].length = read_u4(buffer, buffer_index);
					classinfo->method[index].attribute[index2].array = &buffer[(*buffer_index)];
					(*buffer_index) += classinfo->method[index].attribute[index2].length;
				}
			}
		}
	}

	classinfo->attribute_count = read_u2(buffer, buffer_index);
	if(classinfo->attribute_count > 0) {
		classinfo->attribute = memory_allocate(classinfo->attribute_count, sizeof(AttributeInfo));
		for(index2 = 0; index2 < classinfo->attribute_count; index2++) {
			classinfo->attribute[index2].name_index = read_u2(buffer, buffer_index);
			classinfo->attribute[index2].length = read_u4(buffer, buffer_index);
			classinfo->attribute[index2].array = &buffer[(*buffer_index)];
			(*buffer_index) += classinfo->attribute[index2].length;
		}
	}
}

void classinfo_free(ClassInfo* classinfo) {
	kuint index = 0;

	if(classinfo->constant_count > 0) {
		memory_free(classinfo->constant);
	}

	if(classinfo->interface_count > 0) {
		memory_free(classinfo->interface);
	}

	if(classinfo->field_count > 0) {
		for(index = 0; index < classinfo->field_count; index++) {
			if(classinfo->field[index].attribute_count > 0) {
				memory_free(classinfo->field[index].attribute);
			}
		}
		memory_free(classinfo->field);
	}

	if(classinfo->method_count > 0) {
		for(index = 0; index < classinfo->method_count; index++) {
			if(classinfo->method[index].attribute_count > 0) {
				memory_free(classinfo->method[index].attribute);
			}
		}
		memory_free(classinfo->method);
	}

	if(classinfo->attribute_count > 0) {
		memory_free(classinfo->attribute);
	}

	memory_free(classinfo);
}

void classinfo_print_internal(ClassInfo* classinfo) {
	kuint index = 0;
	kuint index2 = 0;

	print_u4(classinfo->magic);
	print_nl();
	print_u2(classinfo->minor_version);
	print_nl();
	print_u2(classinfo->major_version);
	print_nl();

	print_u2(classinfo->constant_count);
	print_nl();
	for(index = 1; index < classinfo->constant_count; index++) {
		print_sp();
		print_u2(index);
		print_sp();
		print_u1(classinfo->constant[index].info.tag);
		print_sp();
		switch(classinfo->constant[index].info.tag) {
			case CONSTANT_Info_Utf8:
				print_u2(classinfo->constant[index].utf8_info.length);
				print_sp();
				print_u1_array(classinfo->constant[index].utf8_info.bytes, classinfo->constant[index].utf8_info.length);
				print_sp();
				print_utf8(classinfo->constant[index].utf8_info.bytes, classinfo->constant[index].utf8_info.length);
				break;
			case CONSTANT_Info_Integer:
			case CONSTANT_Info_Float:
				print_u4(classinfo->constant[index].v32_info.bytes);
				break;
			case CONSTANT_Info_Long:
			case CONSTANT_Info_Double:
				print_u4(classinfo->constant[index].v64_info.high_bytes);
				print_sp();
				print_u4(classinfo->constant[index].v64_info.low_bytes);
				index++;
				break;
			case CONSTANT_Info_Class:
				print_u2(classinfo->constant[index].class_info.name_index);
				break;
			case CONSTANT_Info_String:
				print_u2(classinfo->constant[index].string_info.string_index);
				break;
			case CONSTANT_Info_FieldRef:
			case CONSTANT_Info_MethodRef:
			case CONSTANT_Info_InterfaceMethodRef:
				print_u2(classinfo->constant[index].ref_info.class_index);
				print_sp();
				print_u2(classinfo->constant[index].ref_info.nametype_index);
				break;
			case CONSTANT_Info_NameAndType:
				print_u2(classinfo->constant[index].nametype_info.name_index);
				print_sp();
				print_u2(classinfo->constant[index].nametype_info.descriptor_index);
				break;
		}
		print_nl();
	}

	print_u2(classinfo->access_flags);
	print_nl();
	print_u2(classinfo->thisclass);
	print_nl();
	print_u2(classinfo->superclass);
	print_nl();

	print_u2(classinfo->interface_count);
	print_nl();
	for(index = 0; index < classinfo->interface_count; index++) {
		print_sp();
		print_u2(index);
		print_sp();
		print_u2(classinfo->interface[index]);
		print_nl();
	}

	print_u2(classinfo->field_count);
	print_nl();
	for(index = 0; index < classinfo->field_count; index++) {
		print_sp();
		print_u2(index);
		print_sp();
		print_u2(classinfo->field[index].access_flags);
		print_sp();
		print_u2(classinfo->field[index].name_index);
		print_sp();
		print_u2(classinfo->field[index].descriptor_index);
		print_sp();
		print_u2(classinfo->field[index].attribute_count);
		print_nl();
		for(index2 = 0; index2 < classinfo->field[index].attribute_count; index2++) {
			print_sp();
			print_sp();
			print_u2(index);
			print_sp();
			print_u2(classinfo->field[index].attribute[index2].name_index);
			print_sp();
			print_u4(classinfo->field[index].attribute[index2].length);
			print_sp();
			print_u1_array(classinfo->field[index].attribute[index2].array, classinfo->field[index].attribute[index2].length);
			print_nl();
		}
	}

	print_u2(classinfo->method_count);
	print_nl();
	for(index = 0; index < classinfo->method_count; index++) {
		print_sp();
		print_u2(index);
		print_sp();
		print_u2(classinfo->method[index].access_flags);
		print_sp();
		print_u2(classinfo->method[index].name_index);
		print_sp();
		print_u2(classinfo->method[index].descriptor_index);
		print_sp();
		print_u2(classinfo->method[index].attribute_count);
		print_nl();
		for(index2 = 0; index2 < classinfo->method[index].attribute_count; index2++) {
			print_sp();
			print_sp();
			print_u2(index);
			print_sp();
			print_u2(classinfo->method[index].attribute[index2].name_index);
			print_sp();
			print_u4(classinfo->method[index].attribute[index2].length);
			print_sp();
			print_u1_array(classinfo->method[index].attribute[index2].array, classinfo->method[index].attribute[index2].length);
			print_nl();
		}
	}

	print_u2(classinfo->attribute_count);
	print_nl();
	for(index2 = 0; index2 < classinfo->attribute_count; index2++) {
		print_sp();
		print_sp();
		print_u2(index);
		print_sp();
		print_u2(classinfo->attribute[index2].name_index);
		print_sp();
		print_u4(classinfo->attribute[index2].length);
		print_sp();
		print_u1_array(classinfo->attribute[index2].array, classinfo->attribute[index2].length);
		print_nl();
	}
}

