/*
 * Copyright (C) 2000 Robert Fitzsimons
 */

#include "types.h"
#include "hashmap.h"
#include "unicode.h"
#include "util.h"

void build_crc32_table();

HashMap* unicode_map;
kuint32* crc32_table;

void init_unicode() {
	unicode_map = hashmap_create();
	build_crc32_table();
}

kuint unicode_put_internal(Unicode* unicode_copy) {
	Unicode* unicode = knull;

	unicode_copy->hash = unicode_hash(unicode_copy->array, unicode_copy->length);
	unicode = hashmap_put(unicode_map, unicode_copy->hash, unicode_copy);

	if(unicode != unicode_copy) {
		memory_free(unicode_copy);
	}

	return unicode->hash;
}

kuint unicode_put(Unicode* unicode) {
	Unicode* unicode_copy = knull;
	kuint size = sizeof(Unicode) + (sizeof(kuint16) * unicode->length);

	unicode_copy = memory_allocate(1, size);
	memory_copy(unicode_copy, unicode, size);

	return unicode_put_internal(unicode_copy);
}

kuint unicode_put_ascii(kuint8* ascii) {
	Unicode* unicode_copy = utf8_to_unicode(ascii, strlen(ascii));
	return unicode_put_internal(unicode_copy);
}

kuint unicode_put_utf8(kuint8* utf8, kuint utf8_length) {
	Unicode* unicode_copy = utf8_to_unicode(utf8, utf8_length);
	return unicode_put_internal(unicode_copy);
}

kuint unicode_put_u2_array(kuint16* u2, kuint u2_length) {
	Unicode* unicode_copy = knull;

	unicode_copy = memory_allocate(1, sizeof(Unicode) + (sizeof(kuint16) * u2_length));
	unicode_copy->length = u2_length;
	memory_copy(unicode_copy->array, u2, (sizeof(kuint16) * u2_length));

	return unicode_put_internal(unicode_copy);
}

Unicode* unicode_get(kuint hash) {
	return hashmap_get(unicode_map, hash);
}

kuint8* unicode_get_ascii(kuint hash) {
	Unicode* unicode = unicode_get(hash);
	kuint ascii_length = 0;
	return unicode_to_utf8(unicode, &ascii_length);
}

kuint8* unicode_get_utf8(kuint hash, kuint* utf8_length) {
	Unicode* unicode = unicode_get(hash);
	return unicode_to_utf8(unicode, utf8_length);
}

Unicode* ascii_to_unicode(kuint8* ascii) {
	return utf8_to_unicode(ascii, strlen(ascii));
}

Unicode* utf8_to_unicode(kuint8* utf8, kuint utf8_length) {
	kuint index = 0;
	kuint length = 0;
	Unicode* unicode = knull;

	while(index < utf8_length) {
		kuint c = utf8[index];
		if(c < 0xC0) {
			index++;
		} else if(c < 0xE0) {
			index += 2;
		} else {
			index += 3;
		}
		length++;
	}

	unicode = memory_allocate(1, sizeof(Unicode) + (sizeof(kuint16) * length));
	unicode->length = length;

	index = 0;
	length = 0;

	while(index < utf8_length) {
		kuint c = utf8[index++];
		kuint cc = 0;
		if(c < 0xC0) {
			cc = c;
		} else if(c < 0xE0) {
			cc = (((c & 0x1F) << 6) + (utf8[index++] & 0x3F));
		} else {
			cc = ((((c & 0x0F) << 12) + ((utf8[index++] & 0x3F) << 6)) + (utf8[index++] & 0x3F));
		}
		unicode->array[length++] = cc;
	}

	return unicode;
}

kuint8* unicode_to_ascii(Unicode* unicode) {
	kuint ascii_length = 0;
	return unicode_to_utf8(unicode, &ascii_length);
}

kuint8* unicode_to_utf8(Unicode* unicode, kuint* utf8_length) {
	kuint index = 0;
	kuint length = 0;
	kuint c = 0;
	kuint8* utf8 = knull;

	for(index = 0; index < unicode->length; index++) {
		c = unicode->array[index];
		if(c < 0x80) {
			length++;
		} else if(c < 0x800) {
			length += 2;
		} else {
			length += 3;
		}
	}

	utf8 = memory_allocate(length + 1, sizeof(kuint8));
	length = 0;

	for(index = 0; index < unicode->length; index++) {
		c = unicode->array[index];
		if(c < 0x80) {
			utf8[length++] = c;
		} else if(c < 0x800) {
			utf8[length++] = (0xC0 | ((c >> 6) & 0x1F));;
			utf8[length++] = (0x80 | ((c >> 0) & 0x3F));;
		} else {
			utf8[length++] = (0xE0 | ((c >> 12) & 0x0F));;
			utf8[length++] = (0x80 | ((c >> 6) & 0x3F));;
			utf8[length++] = (0x80 | ((c >> 0) & 0x3F));;
		}
	}

	utf8[length] = 0;
	(*utf8_length) = length;

	return utf8;
}

void build_crc32_table() {
	kuint32 c = 0;
	kuint32 i = i;
	kuint32 j = 0;

	crc32_table = memory_allocate(256, sizeof(kuint32));

	for(i = 0; i < 256; i++) {
		for(c = i << 24, j = 8; j > 0; --j) {
			c = (c & 0x80000000) ? (c << 1) ^ 0x04C11DB7 : (c << 1);
		}
		crc32_table[i] = c;
	}
}

kuint unicode_hash(kuint16* u2, kuint u2_length) {
	kuint crc = 0xFFFFFFFF;
	kuint array_length = u2_length * sizeof(kuint16);
	kuint8* array = (kuint8*)u2;

	while(array_length-- > 0) {
		crc = (crc << 8) ^ crc32_table[(crc >> 24) ^ *(array++)];
	}

	return ~crc;
}

